///////////////////////////////////////////////////////////////////////  
//  LeafCard.fx
//
//  *** INTERACTIVE DATA VISUALIZATION (IDV) CONFIDENTIAL AND PROPRIETARY INFORMATION ***
//
//  This software is supplied under the terms of a license agreement or
//  nondisclosure agreement with Interactive Data Visualization, Inc. and
//  may not be copied, disclosed, or exploited except in accordance with 
//  the terms of that agreement.
//
//      Copyright (c) 2003-2009 IDV, Inc.
//      All rights reserved in all media.
//
//      IDV, Inc.
//      Web: http://www.idvinc.com


///////////////////////////////////////////////////////////////////////  
//  Global Variables, specific to the leaf card shaders

float4x4    g_mCameraFacingMatrix;


///////////////////////////////////////////////////////////////////////  
//  Shader input/output structures
//
//  The cpp-level source code is setup to feed this vertex format to all of the
//  leaf card vertex shader variations.

struct SLeafCardVertex
{
    float3  m_vPosition         : SPEEDTREE_ATTR0;      // xyz = center pos
    float3  m_vNormal           : SPEEDTREE_ATTR2;      // xyz = normal
    float4  m_vTexCoords        : SPEEDTREE_ATTR8;      // xy = diffuse, zw = lod dimensions
    float4  m_vCorner           : SPEEDTREE_ATTR9;      // xy = corner pos, z = shadow offset, w = planar offset
    float4  m_vTangent          : SPEEDTREE_ATTR10;     // xyz = tangent, w = ambient occlusion
    float4  m_vBinormal         : SPEEDTREE_ATTR11;     // xyz = binormal, w = wind scalar
    float4  m_vWindData         : SPEEDTREE_ATTR12;     // xyzw = wind data
};


///////////////////////////////////////////////////////////////////////  
//  LeafCardVertex

SStandardPixel LeafCardVertex(SLeafCardVertex sIn, uniform bool bUseSpecular)
{
    // this will be fed to the leaf card pixel shader
    SStandardPixel sOut;

    // setup aliases (taken from global variables)
    const float  c_fTreeScale = g_vTreePosAndScale.w;
    const float3 c_vTreePos = g_vTreePosAndScale.xyz;
    const float  c_fLod = g_vTreeRotation.w;
    const float  c_fLodLerp = g_vTreeRotation.z;

    // setup aliases (taken from incoming vertex)
    const float3 c_vCardOffset = sIn.m_vPosition.xyz;
    const float3 c_vNormal = sIn.m_vNormal.xyz;
    const float3 c_vTangent = sIn.m_vTangent.xyz;
    const float3 c_vBinormal = sIn.m_vBinormal.xyz;
    const float  c_fAmbientOcclusion = sIn.m_vTangent.w;
    const float2 c_vDiffuseTexCoords = sIn.m_vTexCoords.xy;
    const float2 c_vCorner = sIn.m_vCorner.xy;
    const float2 c_vLodScales = sIn.m_vTexCoords.zw;
    const float4 c_vWindData = sIn.m_vWindData;
    const float  c_fWindScalar = sIn.m_vBinormal.w;
    const float  c_fPlanarOffset = sIn.m_vCorner.w * c_fTreeScale;

    // compensate for the instance's arbitrary rotation
    float3 vRotatedOffset = AdjustForTreeRotation(c_vCardOffset);
    float3 vRotatedNormal = AdjustForTreeRotation(c_vNormal);
    float3 vRotatedTangent = AdjustForTreeRotation(c_vTangent);
    float3 vRotatedBinormal = AdjustForTreeRotation(c_vBinormal);

    // LOD interpolation; cards are shrunk & grown, depending on the LOD setting
    float fLodScale = lerp(c_vLodScales.y, c_vLodScales.x, c_fLodLerp);
    float2 vLodScaledCorner = c_vCorner * fLodScale;
    float4 vCardCorner = float4(0.0f, vLodScaledCorner, 0.0f);

    // rotate the card to face the camera
    vCardCorner = mul(g_mCameraFacingMatrix, vCardCorner);

    // place the card according to the tree's overall scale
    float3 vPosition = vRotatedOffset + vCardCorner.xyz;
    
    // move the vertex with wind
#ifdef SPEEDTREE_BASIC_WIND
    vPosition = LeafWindMotion(vPosition, vRotatedNormal, c_fWindScalar);
    float3 vWindOffset;
    vPosition = CommonWindMotion(vPosition, c_vWindData, vWindOffset);

    #ifdef SPEEDTREE_WIND_AFFECTS_LIGHTING
        vRotatedNormal += vWindOffset;
        vRotatedNormal = normalize(vRotatedNormal);
        vRotatedBinormal += vWindOffset;
        vRotatedBinormal = normalize(vRotatedBinormal);
        vRotatedTangent += vWindOffset;
        vRotatedTangent = normalize(vRotatedTangent);
    #endif
#endif

    // normal map based lighting
    sOut.m_vNormalVec.x = dot(g_vLightDir, vRotatedTangent);
    sOut.m_vNormalVec.y = dot(g_vLightDir, vRotatedBinormal);
    sOut.m_vNormalVec.z = dot(g_vLightDir, vRotatedNormal);

    // scale the leaf card
    vPosition.xyz *= c_fTreeScale;

    // move the position to this instance's position in the forest
    vPosition.xyz += c_vTreePos;

    // adjust corner to keep all the leaf cards from being coplanar
    vPosition.xyz += c_fPlanarOffset * g_vCameraDirection;

    // compute half vector in tangent space for per-pixel specular lighting effect
    if (bUseSpecular)
        sOut.m_vSpecularHalfVector = ComputeSpecularHalfVector(vRotatedNormal, vRotatedBinormal, vRotatedTangent, vPosition);
    else
        sOut.m_vSpecularHalfVector = float3(0,0,0);

    // final screen projection
    sOut.m_vPosition = ProjectToScreen(vPosition.xyz);

    // compute alpha scalar based on LOD; will cross fade the 3D geometry with billboard images
    float fDistance;
    sOut.m_vDiffuseTexCoords.w = Compute3dFade(fDistance);
    sOut.m_vNormalVec.w = SpecialEffectFade(fDistance);

    // passthrough values
    sOut.m_vDiffuseTexCoords.xy = c_vDiffuseTexCoords;
    sOut.m_vDetailTexCoords.xy = float2(0.0f, 0.0f); // unused in leaf cards
    sOut.m_vDetailTexCoords.w = c_fAmbientOcclusion;

    // shadow projection
#ifdef SPEEDTREE_SHADOWS_ENABLED
    for (int i = 0; i < SPEEDTREE_NUM_SHADOW_MAPS; ++i)
        sOut.m_av2dPosInLightSpace[i] = ProjectToLightSpace(vPosition.xyz, g_amLightViewProjs[i]);

    // need the depth to determine which shadow map to use
    sOut.m_vDetailTexCoords.z = sOut.m_vPosition.z / g_fFarClip;
#else
    sOut.m_vDetailTexCoords.z = 0.0f;
#endif

    // transmission computation
#ifdef SPEEDTREE_TRANSMISSION_LIGHTING
    sOut.m_vDiffuseTexCoords.z = TransmissionBackContribution(vRotatedNormal);
#else
    sOut.m_vDiffuseTexCoords.z = 0.0f;
#endif

    // compute fog value
    sOut.m_vFogData = FogVertex(vPosition.xyz);

    return sOut;
}


///////////////////////////////////////////////////////////////////////  
//  LeafCardVertex_Depth

SStandardPixel_Depth LeafCardVertex_Depth(SLeafCardVertex sIn)
{
    // this will be fed to the leaf card pixel shader
    SStandardPixel_Depth sOut;

    // setup aliases (taken from global variables)
    const float  c_fTreeScale = g_vTreePosAndScale.w;
    const float3 c_vTreePos = g_vTreePosAndScale.xyz;
    const float  c_fLod = g_vTreeRotation.w;
    const float  c_fLodLerp = g_vTreeRotation.z;

    // setup aliases (taken from incoming vertex)
    const float3 c_vCardOffset = sIn.m_vPosition.xyz;
    const float3 c_vNormal = sIn.m_vNormal.xyz;
    const float3 c_vTangent = sIn.m_vTangent.xyz;
    const float3 c_vBinormal = sIn.m_vBinormal.xyz;
    const float  c_fAmbientOcclusion = sIn.m_vTangent.w;
    const float2 c_vDiffuseTexCoords = sIn.m_vTexCoords.xy;
    const float2 c_vCorner = sIn.m_vCorner.xy;
    const float  c_fShadowOffset = sIn.m_vCorner.z;
    const float2 c_vLodScales = sIn.m_vTexCoords.zw;
    const float4 c_vWindData = sIn.m_vWindData;
    const float  c_fWindScalar = sIn.m_vBinormal.w;
    const float  c_fPlanarOffset = sIn.m_vCorner.w * c_fTreeScale;

    // compensate for the instance's arbitrary rotation
    float3 vRotatedOffset = AdjustForTreeRotation(c_vCardOffset);
    float3 vRotatedNormal = AdjustForTreeRotation(c_vNormal);
    
    // LOD interpolation; cards are shrunk & grown, depending on the LOD setting
    float fLodScale = lerp(c_vLodScales.y, c_vLodScales.x, c_fLodLerp);
    float2 vLodScaledCorner = c_vCorner * fLodScale;
    float4 vCardCorner = float4(0.0f, vLodScaledCorner, 1.0f);

    // rotate the card to face the camera
    vCardCorner = mul(g_mCameraFacingMatrix, vCardCorner);

    // place and scale the card according to the tree's overall scale
    float3 vPosition = vRotatedOffset + vCardCorner.xyz;
    
    // move the vertex with wind
#ifdef SPEEDTREE_BASIC_WIND
    vPosition = LeafWindMotion(vPosition, vRotatedNormal, c_fWindScalar);
    float3 vWindOffset;
    vPosition = CommonWindMotion(vPosition, c_vWindData, vWindOffset);
#endif

    // scale the leaf card
    vPosition.xyz *= c_fTreeScale;

    // texcoords
    sOut.m_vDiffuseTexCoords.xy = c_vDiffuseTexCoords;
    sOut.m_vDiffuseTexCoords.z = Compute3dFade( );

    // move the position to this instance's position in the forest
    vPosition.xyz += c_vTreePos;

    // adjust corner to keep all the leaf cards from being coplanar
    vPosition.xyz += c_fPlanarOffset * g_vCameraDirection;

    // final screen projection
    sOut.m_vPosition = ProjectToScreen(vPosition.xyz);

    return sOut;
}


///////////////////////////////////////////////////////////////////////  
//  LeafCardVertex_Shadow

SStandardPixel_Shadow LeafCardVertex_Shadow(SLeafCardVertex sIn)
{
    // this will be fed to the leaf card pixel shader
    SStandardPixel_Shadow sOut;

    // setup aliases (taken from global variables)
    const float  c_fTreeScale = g_vTreePosAndScale.w;
    const float3 c_vTreePos = g_vTreePosAndScale.xyz;
    const float  c_fLod = g_vTreeRotation.w;
    const float  c_fLodLerp = g_vTreeRotation.z;

    // setup aliases (taken from incoming vertex)
    const float3 c_vCardOffset = sIn.m_vPosition.xyz;
    const float3 c_vNormal = sIn.m_vNormal.xyz;
    const float3 c_vTangent = sIn.m_vTangent.xyz;
    const float3 c_vBinormal = sIn.m_vBinormal.xyz;
    const float  c_fAmbientOcclusion = sIn.m_vTangent.w;
    const float2 c_vDiffuseTexCoords = sIn.m_vTexCoords.xy;
    const float2 c_vCorner = sIn.m_vCorner.xy;
    const float  c_fShadowOffset = sIn.m_vCorner.z;
    const float2 c_vLodScales = sIn.m_vTexCoords.zw;
    const float  c_fWindScalar = sIn.m_vBinormal.w;
    const float4 c_vWindData = sIn.m_vWindData;

    // compensate for the instance's arbitrary rotation
    float3 vRotatedOffset = AdjustForTreeRotation(c_vCardOffset);
    float3 vRotatedNormal = AdjustForTreeRotation(c_vNormal);

    // LOD interpolation; cards are shrunk & grown, depending on the LOD setting
    float fLodScale = lerp(c_vLodScales.y, c_vLodScales.x, c_fLodLerp);
    float2 vLodScaledCorner = c_vCorner * fLodScale;
    float4 vCardCorner = float4(0.0f, vLodScaledCorner, 0.0f);

    // rotate the card to face the camera
    vCardCorner = mul(g_mCameraFacingMatrix, vCardCorner);

    // move the offset position away in the light direction to avoid shadowing artifacts
    vRotatedOffset -= c_fShadowOffset * g_vLightDir;

    // place and scale the card according to the tree's overall scale
    float3 vPosition = vRotatedOffset + vCardCorner.xyz;
    
    // move the vertex with wind
#ifdef SPEEDTREE_BASIC_WIND
    vPosition = LeafWindMotion(vPosition, vRotatedNormal, c_fWindScalar);
    float3 vWindOffset;
    vPosition = CommonWindMotion(vPosition, c_vWindData, vWindOffset);
#endif

    vPosition.xyz *= c_fTreeScale;

    // move the position to this instance's position in the forest
    vPosition.xyz += c_vTreePos;

    // final screen projection
    sOut.m_vPosition = ProjectToScreen(vPosition.xyz);

    // compute alpha scalar based on LOD; will cross fade the 3D geometry with
    // billboard images
    sOut.m_vDiffuseTexCoords.z = Compute3dFade( );

    // passthrough values
    sOut.m_vDiffuseTexCoords.xy = c_vDiffuseTexCoords;

    return sOut;
}


///////////////////////////////////////////////////////////////////////  
//  Techniques

SPEEDTREE_TECHNIQUE_TYPE LeafCards
{
    pass P0
    {          
        CompileVS(LeafCardVertex(SPEEDTREE_SPECULAR_LIGHTING));

        CompilePS(StandardPixel(false, // specular effect is expensive and rarely noticed; replace with SPEEDTREE_SPECULAR_LIGHTING if needed
                                 SPEEDTREE_TRANSMISSION_LIGHTING, 
                                 SPEEDTREE_AMBIENT_CONTRAST, 
                                 false, // detail texture always off for leaf cards
                                 false, // detail normal mapping always off for leaf cards
                                 false)); // don't use smooth shadow filters here, too expensive for minimal effect
    }
}

SPEEDTREE_TECHNIQUE_TYPE LeafCards_Depth
{
    pass P0
    {          
        CompileVS(LeafCardVertex_Depth( ));
        CompilePS(StandardPixel_Depth( ));
    }
}

SPEEDTREE_TECHNIQUE_TYPE LeafCards_Shadow
{
    pass P0
    {          
        CompileVS(LeafCardVertex_Shadow( ));
        CompilePS(StandardPixel_Shadow( ));
    }
}


